
# $+HEADER$
# 
# Copyright 2017-2018 Christoph Lueders
# 
# This file is part of the PTCUT project: <http://wrogn.com/ptcut>
# 
# PTCUT is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or 
# (at your option) any later version.
# 
# PTCUT is distributed in the hope that it will be useful, 
# but WITHOUT ANY WARRANTY; without even the implied warranty of 
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
# GNU Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public License
# along with PTCUT.  If not, see <http://www.gnu.org/licenses/>.
# 
# $-HEADER$

from __future__ import print_function, division
from sage.all_cmdline import *   # import sage library

vsn = "1.16.0"
svnrev = int("$Rev: 7016 $".split()[1])
backend = "ppl"
chull_f1 = 2
chull_f2 = 4
chull_f3 = 0.5
chull_f4 = 2

import os
import sys
import time
import numpy as np
from util import *
from biomd import tropicalize_system, load_satyas_solution, biomd_simple, biomd_fast, biomd_slow, biomd_slowhull, biomd_easy, biomd_all, read_grid_data, sample_grid
from prt import *


def generate_polytopes(points, bagnb, verbose=0, complex=False):
   """
   From a dictionary of points with their sign (point is the key, sign the value)
   generate polytopes.  The first coordinate of the keys is the absolute value,
   i.e. an entry of [1,7,3,4] represents the equality 1+7x1+3x2+4x3=0.

   Input: the input list of points.
      Example: points = {(0,6,0): -1, (0,3,1): 1, (0,3,0): -1, (0,1,2): 1}

   Each point is the tropicalization of a monomial.
   The data in the dictionary is +/-1 and contains the sign of that monomial.
   """

   # all points must have the same number of coordinates
   assert all([len(p) == len(points.keys()[0]) for p in points])
   # all entries must have a sign, i.e. none must be zero.
   #assert all([s != 0 for s in points.values()])

   # build a polyhedron of all points
   newton = Polyhedron(vertices=toInt(points.keys()), backend=backend)

   if verbose:
      prt("Points: {}".format(len(points)))
      prt("Dimension of the ambient space: {}".format(len(points.keys()[0])))
      prt("Dimension of the Newton polytope: {}".format(newton.dim()))

      prt("Points on the border of the Newton polytope ({} total):".format(newton.n_Vrepresentation()), flush=True)
      if verbose > 1:
         for i in newton.faces(0):
            prt("   {}: {}".format(i, i.vertices()[0].vector()))

   # cycle through all edges (1-faces) of the boundary of the Newton polytope
   l = []
   cnt = 0
   if verbose:
      prt("Border edges{}:".format("" if complex else " between points with opposing signs"), flush=True)
   for e in newton.faces(1):
      vs = e.vertices()         # get the vertices for the edge
      v1 = vs[0].vector()       # one endpoint of the edge
      v2 = vs[1].vector()       # the other endpoint
      # must convert to tuple, otherwise it's mutable
      if complex or points[tuple(v1)] * points[tuple(v2)] <= 0:
         # opposing sign condition is met
         d = v1 - v2             # edge connecting both vertices
         # build list of inequalities:
         # v_1 = v_2 <= v_i, hence v_i - v_1 >= 0
         ie = []
         for v in newton.faces(0):
            vi = v.vertices()[0]
            if vi != vs[0] and vi != vs[1]:
               ie.append(vi.vector() - v1)
         # sage.geometry.polyhedron.constructor.Polyhedron(vertices=None, rays=None, lines=None,
         #    ieqs=None, eqns=None, ambient_dim=None, base_ring=None, minimize=True, verbose=False,
         #    backend=None)
         # eqns - list of equalities. Each line can be specified as any iterable container of
         #    base_ring elements. An entry equal to [1,7,3,4] represents the equality 1+7x1+3x2+4x3=0
         # ieqs - list of inequalities. Each line can be specified as any iterable container of
         #    base_ring elements. An entry equal to [1,7,3,4] represents the inequality 1+7x1+3x2+4x3>=0
         p = Polyhedron(eqns=[d], ieqs=ie, backend=backend)
         if verbose > 1:
            prt("   {}: {}".format(e, d))
            prt("   eq={}, ieqs={}, dim={}".format(d, ie, p.dim()))
            prt("      {}".format(p.Hrepresentation()), flush=True)
         elif verbose:
            prt("   {}: dim={}".format(e, p.dim()), flush=True)
         if not p.is_empty():        # exclude empty polyhedra
            p.combo = {bagnb: cnt}
            cnt += 1
            l.append(p)
   if not l and len(points) == 1:
      prt("Warning: formula defines no polytope!  Ignored.")
      l = None
   if verbose:
      prt()
   return l


class PtsBag(list):
   def __init__(self, l):
      list.__init__(self, l)
      self.name = ""

def make_polytopes(ts, skip_formula=[], verbose=0, complex=False):
   assert ts
   start = mytime()
   pts_bags = []
   for cnt,t in enumerate(ts):
      prt(end="[{}]".format(cnt), flush=True)
      if verbose:
         prt()
      if cnt in skip_formula:
         prt(end="(skipped)", flush=True)
      else:
         p = generate_polytopes(t, cnt, verbose, complex=complex)
         if p != None:
            p = PtsBag(p)
            p.name = "#" + str(cnt)
            pts_bags.append(p)
   total = mytime() - start
   if not verbose:
      prt()
   prt("Creation time: {:.3f} sec".format(total), flush=True)
   return pts_bags


def var_used_bool(b):
   """
   Measure which variables are used at all in a bag of polytopes.

   >>> p = Polyhedron(eqns=[(0,0,0,4,0,6)], ieqs=[(0,0,2,5,0,4),(0,0,5,2,7,0),(0,0,0,2,0,5)])
   >>> var_used_bool([p])
   [False, True, True, True, True]
   """
   return [bool(i) for i in var_used(b)]

def var_used(b):
   """
   Count the number of uses of a specific variable
   in a bag of polytopes.

   >>> p = Polyhedron(eqns=[(0,0,0,4,0,6)], ieqs=[(0,0,2,5,0,4),(0,0,5,2,7,0),(0,0,0,2,0,5)])
   >>> var_used([p])
   [0.0, 2.0, 4.0, 1.0, 1.0]
   """
   uses = []
   if not b:
      return uses
   for p in b:
      for h in p.Hrepresentation():
         v = h.vector()[1:]
         if len(uses) < len(v):
            uses += [0] * (len(v) - len(uses))     # zero pad
         for cnt,coord in enumerate(v):
            uses[cnt] += coord != 0
   d = 1 / float(len(b))
   return [i*d for i in uses]

def avg_var_used(b):
   """
   Count the number of uses of a specific variable
   in a bag of polytopes.
   """
   uses = []
   cnt = 0
   for p in b:
      for h in p.Hrepresentation():
         mult = 1 if h.is_inequality() else 2
         v = h.vector()[1:]
         if len(uses) < len(v):
            uses += [0] * (len(v) - len(uses))     # zero pad
         for dim,coord in enumerate(v):
            uses[dim] += (coord != 0) * mult
         cnt += mult
   recip = 0 if cnt == 0 else float(1.0 / cnt)
   return [i*recip for i in uses]

import collections
def tuple_cnt(b):
   d = collections.Counter()
   for p in b:
      l = p.n_Hrepresentation() * len(b)
      for h in p.Hrepresentation():
         v = tuple([int(i != 0) for i in h.vector()[1:]])
         d[v] += (0.5 if h.is_inequality() else 1.0) / l
   return d

def hrep_list(b):
   """
   Get a set of H-reps of a bag of polytopes.
   """
   return set([tuple(i.vector()) for p in b for i in p.Hrepresentation()])

def tuple2_cnt(l, w=1):
   d = collections.Counter()
   for h in l:
      v = tuple([int(i != 0) for i in h[1:]])
      d[v] += w
   return d

def tuple3_cnt(p, ln):
   s = [0] * ln
   for h in p:
      for i,c in enumerate(h[1:]):
         if c:
            s[i] += 1
   return s

def likeness(b1, b2, like):
   """
   Calculate a measure of like-ness of two polytope bags.
   """
   if like == 13:
      d = len(b1[0].Hrepresentation()[0]) - 1
      t1, t2 = hrep_list(b1), hrep_list(b2)
      #prt("t1: {}\nt2: {}".format(t1, t2))
      a1, a2 = tuple3_cnt(t1 - t2, d), tuple3_cnt(t2 - t1, d)
      #prt("a1: {}\na2: {}\ncap: {}".format(a1, a2, set(a1.keys()) & set(a2.keys())))
      #return sum([a1[i]*a2[i] for i in set(a1.keys()) & set(a2.keys())])
      lk = sum([min(1.0, u*v) for u,v in zip(a1, a2)]) / (d * len(b1) * len(b2))
      #prt("a1: {}\na2: {}\nlk: {}".format(a1, a2, lk))
      return lk
   if like == 12:
      a1, a2 = tuple_cnt(b1), tuple_cnt(b2)
      lk = 1.0 - prod([(1.0-a1[i]) * (1.0-a2[i]) for i in set(a1.keys()) & set(a2.keys())])
      #prt(a1, a2, lk)
      return lk
   if like == 8:
      # just calculate the average of all likeness'es
      return avg(likeness1_8(p, q) for p in b1 for q in b2)
   if like == 9 or like == 10:
      a1 = avg_var_used(b1)
      a2 = avg_var_used(b2)
      if like == 9:
         return sum(a*b for a,b in zip(a1, a2))
      return 1.0 - prod(1.0 - a*b for a,b in zip(a1, a2))
   if like == 0:
      return 0.0
   if like == 14:
      return -len(b1)
   u1 = var_used(b1)
   u2 = var_used(b2)
   if like == 7:
      return sum([w for v,w in zip(u1, u2) if v*w > 0])
   l1 = sum([min(v,w) != 0 for v,w in zip(u1, u2)])
   l2 = sum([max(v,w) != 0 for v,w in zip(u1, u2)])
   d = len(u1)
   if like == 5:
      return l1 + float(l2)/(d+1)
   if like == 6:
      return l1 + float(d-l2)/(d+1)
   return l1

def likeness1_8(p, q):
   """
   Calculate the likeness of two polytopes.
   Likeness is the probability of the intersection of the two polytopes being empty.
   """
   cnt = 0
   for hp in p.inequalities():
      vp = hp.vector()[1:]
      for hq in q.inequalities():
         vq = hq.vector()[1:]
         cnt += any(u and v for u,v in zip(vp, vq))
   return cnt / (len(p.n_Hrepresentation()) * len(q.n_Hrepresentation()))

def avg_dim(b):
   """
   Return the average dimension of a list of polytopes.
   """
   return sum([p.dim() for p in b]) / float(len(b)) if len(b) > 0 else 0

def avg_planes(b):
   """
   Return the average number of defining hyperplanes of a list of polytopes.
   """
   return sum([p.n_Hrepresentation() for p in b]) / float(len(b)) if len(b) > 0 else 0

def avg_vrep(b):
   """
   Return the average number of defining vertices of a list of polytopes.
   """
   return sum([p.n_Vrepresentation() for p in b]) / float(len(b)) if len(b) > 0 else 0

def avg_vertices(b):
   """
   Return the average number of vertices of a list of polytopes.
   """
   return sum([p.n_vertices() for p in b]) / float(len(b)) if len(b) > 0 else 0

def avg_rays(b):
   """
   Return the average number of rays of a list of polytopes.
   """
   return sum([p.n_rays() for p in b]) / float(len(b)) if len(b) > 0 else 0

def avg_lines(b):
   """
   Return the average number of lines of a list of polytopes.
   """
   return sum([p.n_lines() for p in b]) / float(len(b)) if len(b) > 0 else 0


include_tests = 0
intersections = 0

class NoSolution(Exception):
   pass

class OutOfTime(Exception):
   pass


def np_subpoly(hs, vs, add_const):
   """
   >>> np_subpoly([[0,1,2,3]], [[4,5,6]], False).tolist()
   [[32]]
   >>> np_subpoly([[0,1,2,3],[2,2,2,2]], [[4,5,6]], False).tolist()
   [[32, 30]]
   >>> np_subpoly([[0,1,2,3]], [[4,5,6],[5,6,7]], False).tolist()
   [[32], [38]]
   >>> np_subpoly([[0,1,2,3],[2,2,2,2]], [[4,5,6],[5,6,7]], False).tolist()
   [[32, 30], [38, 36]]
   >>> np_subpoly([[1,1,2,3]], [[4,5,6]], True).tolist()
   [[33]]
   >>> np_subpoly([[1,1,2,3],[2,2,2,2]], [[4,5,6]], True).tolist()
   [[33, 32]]
   >>> np_subpoly([[1,1,2,3],[2,2,2,2]], [[4,5,6],[5,6,7]], True).tolist()
   [[33, 32], [39, 38]]
   >>> np_subpoly([[0,1,0],[0,0,1],[2,-1,-1]], [[0,0],[1,0],[0,1]], True).tolist()
   [[0, 0, 2], [1, 0, 1], [0, 1, 1]]
   >>> np.all(np_subpoly([[0,1,0],[0,0,1],[2,-1,-1]], [[0,0],[1,0],[0,1]], True) >= 0)
   True

   #>>> np_subpoly([[0,1,2,3]], [], False)
   #0
   #>>> np_subpoly([], [4,5,6], False)
   #0
   """
   vs = np.array(vs, dtype=Integer)
   #if vs.shape[0] == 0:
   #   return 0
   assert vs.shape[0] > 0
   hs = np.array(hs, dtype=Integer)
   #if hs.shape[0] == 0:
   #   return 0
   assert hs.shape[0] > 0
   assert hs.shape[1] - 1 == vs.shape[1]
   const = hs[:,:1].T                        # snip off constants
   hs = hs[:,1:]                             # snip off equations
   inner = np.inner(hs, vs).T
   if add_const:
      inner += const
   return inner


def my_is_subpolyhedron(sub, p):
   """
   Return if polyhedron sub is contained in polyhedron p.

   >>> p = Polyhedron(vertices=[[0,0],[2,0],[0,2]])      # H-rep is 3 inequalities
   >>> my_is_subpolyhedron(Polyhedron(vertices=[[0,0],[1,0],[0,1]]), p)
   True
   >>> my_is_subpolyhedron(Polyhedron(vertices=[[0,0],[1,0],[0,3]]), p)
   False
   >>> my_is_subpolyhedron(Polyhedron(vertices=[[0,0],[2,0],[0,2]]), p)
   True

   >>> p = Polyhedron(eqns=[[-1,1,0,0],[-1,0,1,0]])      # 1 vertex and 1 line, x1=1, x2=1, x3 free
   >>> my_is_subpolyhedron(Polyhedron(vertices=[[1,1,2]]), p)
   True
   >>> my_is_subpolyhedron(Polyhedron(vertices=[[1,0,2]]), p)
   False
   >>> my_is_subpolyhedron(Polyhedron(vertices=[[0,1,2]]), p)
   False

   >>> p = Polyhedron(ieqs=[[-1,1,0,0],[-1,0,1,0]])      # 1 vertex and 1 line, 2 rays, x1>=1, x2>=1, x3 free
   >>> my_is_subpolyhedron(Polyhedron(vertices=[[1,1,2]]), p)
   True
   >>> my_is_subpolyhedron(Polyhedron(vertices=[[1,2,2]]), p)
   True
   >>> my_is_subpolyhedron(Polyhedron(vertices=[[2,1,2]]), p)
   True
   >>> my_is_subpolyhedron(Polyhedron(vertices=[[1,0,2]]), p)
   False
   >>> my_is_subpolyhedron(Polyhedron(vertices=[[0,1,2]]), p)
   False

   # p has 4 equations, 7 inequalities
   # sub has 3 rays, 2 vertices
   >>> p = Polyhedron(eqns=[[-4, 0, 0, 0, 1, 0, 1, 0, -1], [3, 1, 0, 0, -1, 0, -1, 1, 0], [3, 1, 0, 0, -1, 1, 0, 0, 0], [3, 1, 1, -1, 0, 0, 0, 0, 0]], ieqs=[[3, -1, 0, 0, 0, 0, 1, 0, 0], [1, 0, 0, 0, 0, 0, 1, 0, 0], [0, -1, 0, 0, 1, 0, 0, 0, 0], [-2, -1, 0, 0, 1, 0, 1, 0, 0], [3, 1, 1, 0, -1, 0, 0, 0, 0], [-4, 0, 0, 0, 1, 0, 1, 0, 0], [0, 1, 0, 0, 0, 0, 0, 0, 0]])
   >>> sub = Polyhedron(eqns=[[-7, 1, 0, 0, 1, 0, 0, 0, -1], [-6, 0, 0, 0, 1, 0, 0, -1, 0], [-3, 1, 0, 0, 0, 0, -1, 0, 0], [3, 1, 0, 0, -1, 1, 0, 0, 0], [3, 1, 1, -1, 0, 0, 0, 0, 0]], ieqs=[[3, 1, 1, 0, -1, 0, 0, 0, 0], [-2, 1, 0, 0, 0, 0, 0, 0, 0], [0, -1, 0, 0, 1, 0, 0, 0, 0], [-5, 0, 0, 0, 1, 0, 0, 0, 0]])
   >>> my_is_subpolyhedron(sub, p)
   True

   To check against an equation:
      the inner product of all V-objects (vertices, rays, lines) with the equation must be zero.
   To check against an inequality:
      the inner product of vertices and rays with the inequality must be non-negative,
      the inner product of lines with the inequality must be zero.
   """
   vertices = sub.vertices_list()
   rays = sub.rays_list()
   lines = sub.lines_list()
   eq = p.equations_list()
   if eq:
      if not all(np.all(np_subpoly(eq, [v], True) == 0) for v in vertices):
         return False
      #if vertices and not np.all(np_subpoly(eq, vertices, True) == 0):
      #   return False
      if not all(np.all(np_subpoly(eq, [v], False) == 0) for v in rays):
         return False
      #if rays and not np.all(np_subpoly(eq, rays, False) == 0):
      #   return False
      if not all(np.all(np_subpoly(eq, [v], False) == 0) for v in lines):
         return False
      #if lines and not np.all(np_subpoly(eq, lines, False) == 0):
      #   return False
   ie = p.inequalities_list()
   if ie:
      if not all(np.all(np_subpoly(ie, [v], True) >= 0) for v in vertices):
         return False
      #if vertices and not np.all(np_subpoly(ie, vertices, True) >= 0):
      #   return False
      if not all(np.all(np_subpoly(ie, [v], False) >= 0) for v in rays):
         return False
      #if rays and not np.all(np_subpoly(ie, rays, False) >= 0):
      #   return False
      if not all(np.all(np_subpoly(ie, [v], False) == 0) for v in lines):
         return False
      #if lines and not np.all(np_subpoly(ie, lines, False) == 0):
      #   return False
   return True


def is_subpolyhedron(r, i):
   # return r <= i
   # does internally one check too much, see
   # src\sage\geometry\polyhedron\base.py: __cmp__(self, other), line 384
   return r._is_subpolyhedron(i)
   
   if False:
      for other_H in i.Hrepresentation():
         for self_V in r.Vrepresentation():
            ret = other_H.contains(self_V)
            prt("H {} contains V {} = {}".format(other_H.vector(), self_V.vector(), ret), log=True, screen=False)
            if not ret:
               return False
      return True

   t = mytime()
   r1 = r._is_subpolyhedron(i)
   t = mytime() - t
   prtei = lambda x: str([tuple(i) for i in x]).replace(" ", "")
   s = \
"""   (
      # time= {} hs= {} vs= {}
      NNN, {}, {}, {}, {}, {},
      {},
      {},
      {},
      {},
   ),
""".format(
      t, r.n_Hrepresentation() + i.n_Hrepresentation(), r.n_Vrepresentation() + i.n_Vrepresentation(),
      i.n_Hrepresentation(), 
      i.n_Vrepresentation(),
      r.n_Hrepresentation(),
      r.n_Vrepresentation(), 
      r1,
      prtei(i.equations_list()),
      prtei(i.inequalities_list()),
      prtei(r.equations_list()),
      prtei(r.inequalities_list()))
   prt(s, screen=False)
   return r1

   r2 = my_is_subpolyhedron(r, i)
   return r2
   if r1 != r2:
      prt("ieq={}\niie={}\nreq={}\nrie={}, right={}, wrong={}\n".format(
         i.equations_list(), i.inequalities_list(), r.equations_list(), r.inequalities_list(), r1, r2))
      sys.exit(1)
   return r1


def insert_include(l, n):
   """
   Insert polytope n into list l, but check that n is not included
   in any of l's members and that n doesn't include any of l's members.
   If so, use the larger one.
   If both are equal, keep the one with the smaller combo index.
   """
   global include_tests
   strict_order = False
   # we're assuming that it is more likely that the new polyhedron is already included in the list,
   # so we check for that first.
   for idx,i in enumerate(l):
      # is n contained in i, i.e. something we already know?
      if is_subpolyhedron(n, i):
         if strict_order and is_subpolyhedron(i, n):
            # both are included in one another, i.e. they are equal.  keep the one with the lower combo index
            if combo_cmp(i.combo, n.combo) < 0:
               l.insert(0, l.pop(idx))    # move to old i front of list
            else:
               l.pop(idx)                 # remove old i with larger combo index
               l.insert(0, n)             # put new n to front of list
         else:
            # i is strictly larger than n
            l.insert(0, l.pop(idx))       # move to front of list
         include_tests += 1 + int(strict_order)
         return l                         # no need to continue
      include_tests += 1
   # n is not included in any of l's polyhedra, so it has to be added to the list anyway.
   # see if n includes any of the already existing ones.
   l2 = []
   for i in l:
      # is i contained in n?  if so, don't copy it to new list.
      if not is_subpolyhedron(i, n):
         # here, all inclusions are strict, otherwise they would have been found in pass 1.
         l2.append(i)                     # append i to new list
      include_tests += 1
   l2.append(n)
   return l2


def myintersect(p, q):
   # converting to set will remove double mention of planes
   peq = set((tuple(i) for i in p.equations_list()))
   pie = set((tuple(i) for i in p.inequalities_list()))
   qeq = set((tuple(i) for i in q.equations_list()))
   qie = set((tuple(i) for i in q.inequalities_list()))
   #t = mytime()
   r = Polyhedron(eqns=peq | qeq, ieqs=pie | qie, backend=backend)
   #t = mytime() - t
   #prt("peq={}\npie={}\nqeq={}\nqie={}\ntime={} hs={} vs={}\n".format(list(peq), list(pie), list(qeq), list(qie), t, r.n_Hrepresentation(), r.n_Vrepresentation()).replace(" ", ""), screen=False)
   r.combo = p.combo
   r.combo.update(q.combo)
   return r


def intersect_two(b1, b2, verbose=0, text="", like=9, endtime=inf):
   """
   Intersect all elements of p with all elements of q.
   Only save non-empty results and make sure that no element
   of the resulting list is included in any other element of it.
   """
   if verbose:
      prt(end="[{}({}, {})]: {} * {} = {}".format(text, b1.name, b2.name, len(b1), len(b2), len(b1)*len(b2)), flush=True)
   # from the two bags, take all combinations of one polytope each and intersect those combinations.
   # if the intersection is empty, drop it.
   # if the intersection is contained in some other intersection we already have, drop it.
   # if the intersection is a superset of some other intersection we already have, drop that other intersection
   #   and add the new intersection.
   l = []                           # new list of polytopes
   empty_cnt = 0
   cnt = 0
   tame = tame_it(2)
   sts = status_print()
   global include_tests
   global intersections
   if mytime() > endtime:
      raise OutOfTime
   for p in b1:
      for q in b2:
         cnt += 1
         if tame.go():
            if mytime() > endtime:
               raise OutOfTime
            sts.print("({})".format(cnt))
         #r = p & q                  # intersect
         r = myintersect(p, q)
         intersections += 1
         if r.is_empty():          # ignore if the intersection is empty
            empty_cnt += 1
            continue
         l = insert_include(l, r)
   sts.print("")
   if verbose:
      lk = likeness(b1, b2, like)
      lstr = "" if like in (0,14) else " like={:.2f}%".format(lk * 100.0) if like in (10,12,13) else " like={:.1f}".format(float(lk))
      prt(" => {} ({} empty, {} incl){}, dim={:.1f}, used={}, hs={:.1f}, vs={:.1f}".format(
         len(l), empty_cnt, len(b1)*len(b2) - len(l) - empty_cnt,
         lstr, avg_dim(l), sum(var_used_bool(b2)), avg_planes(l), avg_vrep(l)), flush=True, flushfile=True)
   if len(l) == 0:
      raise NoSolution
   return PtsBag(l)


def highest_likeness(bags, like):
   minlike = [10]
   lmax = inf if like in minlike else -inf
   for i,b in enumerate(bags):
      for j,c in enumerate(bags):
         if j >= i:
            break
         lk = likeness(b, c, like)
         if like in minlike:
            #lk = max(0.0, lk)
            #lk = min(1.0, lk)
            #prt("  {} {} {} {} {} {}".format(b.name, c.name, len(b), len(c), lk, lk * len(b) * len(c)))
            cmp = (1.0 - lk) * len(b) * len(c) < lmax
         else:
            cmp = lk > lmax
         if cmp:
            lmax = lk
            maxij = (i ,j)
   j = maxij[0]
   i = maxij[1]
   assert j > i
   bj = bags[j]
   bags.pop(j)
   bi = bags[i]
   bags.pop(i)
   bags.insert(0, bj)
   bags.insert(1, bi)


# eqns - list of equalities. An entry equal to [1,7,3,4] represents the equality 1+7x1+3x2+4x3=0
# ieqs - list of inequalities. An entry equal to [1,7,3,4] represents the inequality 1+7x1+3x2+4x3>=0
def mkdict(l, add_neg=False):
   """
   Convert a list of vectors (lists) to a dict (the 0-th coeff as value, the rest as key),
   optionally adding the negatives of those vectors as well since x = 2 implies x >= 2 as well as x <= 2).

   >>> sorted(mkdict([[1,2,3],[-4,-5,-6]]).items())
   [((-5, -6), -4), ((2, 3), 1)]
   >>> sorted(mkdict([[1,2,3],[-4,-5,-6]], True).items())
   [((-5, -6), -4), ((-2, -3), -1), ((2, 3), 1), ((5, 6), 4)]
   """
   d = {}
   for v in l:
      v0 = v[0]                  # the absolute value
      vx = tuple(v[1:])          # the normal vector
      assert vx not in d
      d[vx] = v0
      if add_neg:
         # negate, i.e. from 7x1+5x2 >= 2 generate 7x1+5x2 <= 2 as well
         vx = tuple([-i for i in vx])        
         assert vx not in d
         d[vx] = -v0
   return d
   

def dict_to_set(d):
   """
   Convert the normal vector/absolute dicts back to sets.

   >>> sorted(dict_to_set({(2,3):1, (5,6):4}))
   [(1, 2, 3), (4, 5, 6)]
   """
   s = set()
   for k,v in d.items():
      s.add(tuple([v] + list(k)))
   return s
   

def join_dict(a, b):
   """
   >>> sorted(join_dict({1:2, 2:4}, {3:5, 4:9}).items())
   [(1, 2), (2, 4), (3, 5), (4, 9)]
   """
   d = a.copy()
   d.update(b)
   return d


def common_restrictions(b):
   """
   Find the common restrictions for a list of polyhedra.
   
   >>> p = Polyhedron(eqns=[(1,2,3)], ieqs=[(4,5,6)])
   >>> q = Polyhedron(eqns=[(1,2,2)], ieqs=[(4,5,7)])
   >>> common_restrictions([p, q])
   (set([]), set([]))
   
   >>> p = Polyhedron(eqns=[], ieqs=[(4,5,6)])
   >>> q = Polyhedron(eqns=[], ieqs=[(4,5,6)])
   >>> common_restrictions([p, q])
   (set([]), set([(4, 5, 6)]))

   >>> p = Polyhedron(eqns=[(1,2,3)], ieqs=[])
   >>> q = Polyhedron(eqns=[(2,2,3)], ieqs=[])
   >>> [sorted(s) for s in common_restrictions([p, q])]
   [[], [(-1, -2, -3), (2, 2, 3)]]

   >>> p = Polyhedron(eqns=[], ieqs=[(1,2,3)])
   >>> q = Polyhedron(eqns=[], ieqs=[(2,2,3)])
   >>> common_restrictions([p, q])
   (set([]), set([(2, 2, 3)]))
   """
   assert len(b) >= 2
   # initialize with first element of b
   comeq = mkdict(b[0].equations_list())
   comie = join_dict(mkdict(b[0].inequalities_list()), mkdict(b[0].equations_list(), True))
   assert len(comie) == b[0].n_inequalities() + b[0].n_equations()*2    # there should be no overlap
   # loop over the rest
   for p in b[1:]:
      eq = mkdict(p.equations_list())
      ie = join_dict(mkdict(p.inequalities_list()), mkdict(p.equations_list(), True))
      assert len(ie) == p.n_inequalities() + p.n_equations()*2          # there should be no overlap
      # check still common inequalities 
      for k,v in comie.copy().items():
         if k not in ie:
            del comie[k]                              # it's not in both, i.e. delete in comie
         elif ie[k] != v:
            comie[k] = max(v, ie[k])                  # if comie says x-2 >= 0 (x >= 2) and ie says x-4 >= 0 (x >= 4), then surely x-2 >= 0 (x >= 2)
      # check still common equations 
      for k,v in comeq.copy().items():
         if k not in eq:
            del comeq[k]                              # it's not in both, i.e. delete in comeq
         elif eq[k] != v:
            comie[k] = max(comie[k], eq[k])           # if comeq says x-2 = 0 (x = 2) and eq says x-4 = 0 (x = 4), then surely x-2 >= 0 (x >= 2)
            del comeq[k]                              # no longer an equation
   # convert back
   return dict_to_set(comeq), dict_to_set(comie)


def mkset(l, add_neg=False):
   """
   Convert a list of vectors (lists) to a set,
   optionally adding the negatives of those vectors as well.

   >>> sorted(mkset([[1,2,3],[-4,-5,-6]]))
   [(-4, -5, -6), (1, 2, 3)]
   >>> sorted(mkset([[1,2,3],[-4,-5,-6]], True))
   [(-4, -5, -6), (-1, -2, -3), (1, 2, 3), (4, 5, 6)]
   """
   s = set()
   for v in l:
      s.add(tuple(v))
      if add_neg:
         s.add(tuple([-i for i in v]))
   return s


common_plane_time = 0
hull_time = 0

import random
def common_planes(pts_bags, old_common, verbose=False, convexhull=True, endtime=inf):
   start = mytime()
   global common_plane_time, hull_time
   #prt("Finding common planes... ")
   combi = prod([len(i) for i in pts_bags])
   old_len = len(pts_bags)
   global intersections
   oeq, oie, oldc = old_common
   worked = False
   loop = True
   debug_hull = False
   max_planes = 5000

   # loop will run as long as we find new constraints
   while loop:
      if mytime() > endtime:
         raise OutOfTime
      common_eq = set()
      common_ie = set()
      hull_eq = set()
      hull_ie = set()
      one_eq = set()
      one_ie = set()
      one_cnt = 0
      one_combo = {}
      for cnt,b in enumerate(pts_bags):
         # make sure b.hull exists
         if "hull" not in b.__dict__:
            b.hull = False
         if len(b) == 1:
            # bags with just one polytope
            one_eq.update(mkset(b[0].equations_list()))
            one_ie.update(mkset(b[0].inequalities_list()))
            one_cnt += 1
            one_combo.update(b[0].combo)
         elif b:
            # find common (in)equalities between all polyhedra.
            beq, bie = common_restrictions(b)
            if convexhull:
               vertices = set()
               rays = set()
               lines = set()
               for p in b:
                  vertices.update(mkset(p.vertices_list()))
                  rays.update(mkset(p.rays_list()))
                  lines.update(mkset(p.lines_list()))
            common_eq.update(beq)
            common_ie.update(bie)
            # build convex hull of all polytopes of this bag
            b.hull = False
            if convexhull and len(vertices)+len(rays)+len(lines) <= max(avg_dim(b) * chull_f1, avg_planes(b) * chull_f3):
               if debug_hull:
                  prt(end="[hull {}v".format(len(vertices)+len(rays)+len(lines)), flush=True)
               start_hull = mytime()
               hull = Polyhedron(vertices=vertices, rays=rays, lines=lines, backend=backend)
               hull_time += mytime() - start_hull
               assert not hull.is_empty()
               hull_eq.update(mkset(hull.equations_list()))
               hull_ie.update(mkset(hull.inequalities_list()))
               b.hull = True
               if debug_hull:
                  prt(end=" {}h]".format(hull.n_Hrepresentation()), flush=True)

      # remove already known planes, but not from one_*, since they are filtered out later
      common_eq -= oeq
      hull_eq -= oeq
      common_ie -= oie
      hull_ie -= oie

      # make sure we're not using too many planes
      # we're only limiting the number of inequalities.
      n = max_planes
      if convexhull:
         n = min(n, int(avg_dim(b) * chull_f2))
      xie = common_ie | hull_ie
      if len(xie) > n:
         if debug_hull:
            prt(end="[sample {} {}]".format(n, len(xie)), flush=True)
         xie = set(random.sample(xie, n))
      # any new planes?
      eq = one_eq | common_eq | hull_eq
      ie = one_ie | xie
      if not eq - oeq and not ie - oie:
         break
      oeq |= eq
      oie |= ie
      # build a polyhedron of all planes
      if debug_hull:
         prt(end="[cut {}h {} {} {} {} {} {}".format(len(eq)+len(ie), len(one_eq), len(common_eq), len(hull_eq), len(one_ie), len(common_ie), len(hull_ie)), flush=True)
      assert eq | ie
      c = Polyhedron(eqns=eq, ieqs=ie, backend=backend)
      if debug_hull:
         prt(end=" {}v]".format(c.n_Vrepresentation()), flush=True)
      if c.is_empty():
         # empty polyhedron, i.e. no solution
         common_plane_time += mytime() - start
         raise NoSolution
      if convexhull:
         hull_ok = True
         # test against polyhedron without hull
         if debug_hull:
            prt(end="[recut {}h".format(len(eq)+len(ie)), flush=True)
         c2 = Polyhedron(eqns=eq, ieqs=common_ie | one_ie, backend=backend)
         if debug_hull:
            prt(end=" {}v]".format(c.n_Vrepresentation()), flush=True)
         if c.n_Vrepresentation() > c2.n_Vrepresentation() * chull_f4:
            hull_ok = False
            c = c2
            if debug_hull:
               prt(end="[not used]", flush=True)
      c.combo = one_combo
      # is new c any more restrictive?
      if oldc is not None and is_subpolyhedron(oldc, c):
         break
      oldc = c

      loop = False
      worked = True
      bags = []
      cnt = 1
      prt("Applying common planes (dim={}, hs={}, vs={})...".format(c.dim(), c.n_Hrepresentation(), c.n_Vrepresentation()))
      for b in pts_bags:
         empty_cnt = 0
         if len(b) != 1:
            prt(end=" [{}/{} ({})]: {}".format(cnt, len(pts_bags) - one_cnt, b.name, len(b)), flush=True)
            cnt += 1
            b2 = []
            pcnt = 0
            tame = tame_it(2)
            sts = status_print()
            for p in b:
               pcnt += 1
               if tame.go():
                  if mytime() > endtime:
                     raise OutOfTime
                  sts.print("({})".format(pcnt))
               r = myintersect(p, c)
               intersections += 1
               if r.is_empty():
                  empty_cnt += 1
                  continue
               b2 = insert_include(b2, r)
            sts.print("")
            b2 = PtsBag(b2)
            b2.name = b.name
            bags.append(b2)
            loop |= len(b2) == 1       # loop if resulting bag has only one polytope
            incl_cnt = len(b) - len(b2) - empty_cnt
            prt(" => {} ({} empty, {} incl), dim={:.1f}, used={}, hs={:.1f}, vs={:.1f}{}".format(len(b2), empty_cnt, incl_cnt,
               avg_dim(b2), sum(var_used_bool(b2)), avg_planes(b2), avg_vrep(b2), " (hull)" if b.hull and hull_ok else ""), flush=True)
            if len(b2) == 0:
               common_plane_time += mytime() - start
               raise NoSolution
      pts_bags = bags
      if not pts_bags:
         break
   # filter bags with just one polytope.  their constraits have already been intersected with the other bags.
   # this might happen, since the intersection only starts if eq and ie and c contains something new.
   pts_bags = [p for p in pts_bags if len(p) != 1]
   # make sure one_combo is not lost for the same reasons
   if not pts_bags:
      # if no bags left, return the common intersection c
      pts_bags = [PtsBag([c])]
      pts_bags[0].name = "c"
   for p in pts_bags[0]:
      p.combo.update(one_combo)
   if worked:
      prd = prod([len(i) for i in pts_bags])
      combi = combi // prd if combi / prd >= 100 else combi / prd
      prt(" Savings: factor {:,}, {} formulae".format(combi, old_len - len(pts_bags)), flush=True)
   common_plane_time += mytime() - start
   return pts_bags, (oeq, oie, oldc)


def intersect_all(pts_bags, verbose=0, algo=0, like=9, common=True, sorting=0, resort=False, convexhull=True, fname1=None, max_run_time=inf):
   """
   Calculate the intersections of one polytope per bag.

   Input: 'pts_bags' is a list of lists of polytopes ("bags of polytopes").

   Output: a list of polytopes.

   Take all combinations of one polytope per "bag".
   Intersect the polytopes of each combination.
   If, in the result, one polytope is included in another, drop the included one from the solution set.
   """
   stats = {"max_pt": 0, "aborted": False}
   if not pts_bags:                    # no input
      return [], stats
   combis = prod(len(i) for i in pts_bags)
   old_common = set(), set(), None
   if verbose:
      dim = -2
      for i in pts_bags:
         if i:
            dim = len(i[0].Hrepresentation()[0].vector())-1
            break
      if dim != -2:
         prt("Dimension: {}".format(dim))
         prt("Formulae: {}".format(len(pts_bags)))
         prt(end="Order:")
         for i in pts_bags:
            prt(end=" {}".format(i.name[1:]))
         prt()
         prt(end="Worst case combinations: ")
         for cnt,i in enumerate(pts_bags):
            prt(end="{}{} ".format("* " if cnt > 0 else "", len(i)))
         prt(" = {:,}".format(combis), flush=True)
   if fname1:
      fname1 += "-temp"
   global intersections, include_tests, common_plane_time, hull_time
   intersections = 0
   include_tests = 0
   common_plane_time = 0
   hull_time = 0
   start = mytime()
   ftime = 0
   endtime = mytime() + max_run_time

   try:
      if algo == 2:
         prt("Looking for best start bag:"),
         lmax = -1
         for i,b in enumerate(pts_bags):
            for j,c in enumerate(pts_bags):
               if j > i:
                  lk = likeness(b, c, like)
                  if lk > lmax:
                     lmax = lk
                     maxij = (i, j)
         m = maxij[0]
         prt(pts_bags[m].name)
         pts_bags = [pts_bags[m]] + pts_bags[:m] + pts_bags[m+1:]
      run = 1
      runs = len(pts_bags) - 1
      work_cnt = 1
      while len(pts_bags) > 1:
         if common:
            # find common planes and apply them to all bags
            old = len(pts_bags)
            pts_bags, old_common = common_planes(pts_bags, old_common, verbose=verbose, convexhull=convexhull and run == 1, endtime=endtime)
            if len(pts_bags) <= 1:
               break
            runs -= old - len(pts_bags)
            # if resorting is on, keep it sorted in every step
            if resort or run == 1:
               if sorting == 1:
                  pts_bags = sorted(pts_bags, key=lambda x: len(x))
               elif sorting == 2:
                  pts_bags = sorted(pts_bags, key=lambda x: -len(x))

         if algo == 0:
            # standard breadth first
            l = intersect_two(pts_bags[0], pts_bags[1], verbose, text="{}/{} ".format(run, runs), like=like, endtime=endtime)
            pts_bags = [l] + pts_bags[2:]
         elif algo == 1:
            # join and conquer
            i = 0
            bags2 = []
            while i < len(pts_bags) - 1:
               b1 = pts_bags[i]
               b2 = pts_bags[i+1]
               l = intersect_two(b1, b2, verbose, like=like, text="{}: ".format(run), endtime=endtime)
               bags2.append(l)
               i += 2
            # odd number of bags?
            if i < len(pts_bags):
               bags2.append(pts_bags[i])
            pts_bags = bags2
         elif algo == 2:
            # find the next one that has the highest likeness
            lmax = -1
            nhmin = 2**63
            if like == 9 and True:
               gavg = combis ** (1.0 / len(pts_bags))
               if gavg > len(pts_bags[0]):
                  bags2 = [i for i in pts_bags[1:] if len(i) >= gavg]
                  bags1 = [i for i in pts_bags[1:] if not len(i) >= gavg]
               else:
                  bags2 = [i for i in pts_bags[1:] if len(i) <= gavg]
                  bags1 = [i for i in pts_bags[1:] if not len(i) <= gavg]
               #prt(gavg, [len(p) for p in bags2])
            else:
               bags1 = []
               bags2 = pts_bags[1:]
            for b in bags2[:]:
               lk = likeness(pts_bags[0], b, like)
               nh = avg_planes(b)
               bags2 = bags2[1:]
               if lk > lmax or (like == 3 and lk == lmax and nh < nhmin):
                  lmax = lk
                  nhmin = nh
                  b1max = bags1[:]
                  b2max = bags2[:]
                  bmax = b
               bags1.append(b)
            l = intersect_two(pts_bags[0], bmax, verbose, like=like, text="{}/{} ".format(run, runs), endtime=endtime)
            l.name = "w"
            combis = combis / (len(pts_bags[0]) * len(bmax)) * len(l)
            pts_bags = [l] + b1max + b2max
         elif algo == 3:
            # find the next pair that has the highest likeness
            highest_likeness(pts_bags, like)       # sort the two with the highest likeness to the front
            l = intersect_two(pts_bags[0], pts_bags[1], verbose, like=like, text="{}/{} ".format(run, runs), endtime=endtime)
            l.name = "w{}".format(work_cnt)
            work_cnt += 1
            combis = combis / (len(pts_bags[0]) * len(pts_bags[1])) * len(l)
            pts_bags = [l] + pts_bags[2:]
         run += 1
         stats["max_pt"] = max(stats["max_pt"], len(pts_bags[0]))
         if fname1:
            save_polytopes(pts_bags, fname1, quiet=True)
      assert len(pts_bags) == 1

   except NoSolution:
      pts_bags = [PtsBag([])]
   except OutOfTime:
      prt("\n****** Computation aborted!!!  Out of time!!! ******")
      stats["aborted"] = True
      pts_bags = [PtsBag([])]

   stats["max_pt"] = max(stats["max_pt"], len(pts_bags[0]))
   total = mytime() - start - ftime
   prt("Intersection time: {:.3f} sec, total intersections: {:,}, total inclusion tests: {:,}{}".format(
      total, intersections, include_tests, " (aborted)" if stats["aborted"] else ""), flush=True)
   if common:
      prt("Common plane time: {:.3f} sec{}".format(common_plane_time, ", convex hull time: {:.3f} sec".format(hull_time) if convexhull else ""), flush=True)
   if fname1:
      try:
         os.remove(fname1 + "-polytopes.txt")
      except OSError:
         pass
   return pts_bags[0], stats


def _is_connected(p, q, typ):
   """
   >>> _is_connected(Polyhedron(vertices=[[0,0],[0,1]]), Polyhedron(vertices=[[0,1],[0,2]]), typ=0)
   True
   >>> _is_connected(Polyhedron(vertices=[[0,0],[0,1]]), Polyhedron(vertices=[[0,1],[0,2]]), typ=1)
   True
   >>> _is_connected(Polyhedron(vertices=[[0,0],[0,1]]), Polyhedron(vertices=[[0,3],[0,2]]), typ=0)
   False
   >>> _is_connected(Polyhedron(vertices=[[0,0],[0,1]]), Polyhedron(vertices=[[0,3],[0,2]]), typ=1)
   False
   >>> _is_connected(Polyhedron(vertices=[[0,0],[0,1],[1,1]]), Polyhedron(vertices=[[0,1],[0,2],[1,2]]), typ=0)
   True
   >>> _is_connected(Polyhedron(vertices=[[0,0],[0,1],[1,1]]), Polyhedron(vertices=[[0,1],[0,2],[1,2]]), typ=1)
   False
   >>> _is_connected(Polyhedron(vertices=[[0,0],[0,1],[1,1]]), Polyhedron(vertices=[[0,1],[0,2],[1,1]]), typ=1)
   True
   """
   c = p & q
   return is_connected(p, q, c, typ)

def is_connected(p, q, c, typ):
   if c.is_empty():
      return False
   # there is some connection
   if typ == 0:
      return True
   if typ == 2:
      return p.dim() == q.dim()
   mindim = min(p.dim(), q.dim())
   if mindim == 0:
      return True
   return c.dim() >= mindim - 1


import itertools
def build_graph(b, con_type, dbg=False):
   """
   build adjacency matrix of a bag of polytopes.
   an entry is True, iff the two polytopes are connected (i.e. the intersection is not empty)
   """
   # init
   s = ""
   l = len(b)
   adj = np.zeros((l, l), dtype=bool)
   for i,p in enumerate(b):
      p.idx = i
   # test two nodes for connection
   for p,q in itertools.combinations(b, 2):
      c = p & q
      if is_connected(p, q, c, con_type):
         if dbg:
            prt("Connection between {} and {}, dim={}/{}/{}".format(p.idx, q.idx, p.dim(), q.dim(), c.dim()))
         adj[p.idx, q.idx] = True
         adj[q.idx, p.idx] = True
         s += "{}:{}:{}/{}/{}\n".format(p.idx, q.idx, p.dim(), q.dim(), c.dim())
   for p in b:
      if dbg and not any(adj[p.idx,:]):
         prt("No connection to {}, dim={}".format(p.idx, p.dim()))
   return adj, s


def status(time, file=sys.stdout):
   print("\n" + "-" * 70, file=file)
   print(version(), file=file)
   print("Python {}".format(sys.version), file=file)
   import util
   import biomd
   import graph
   import fuse
   import prt
   r = max(biomd.svnrev, fuse.svnrev, graph.svnrev, prt.svnrev, svnrev, util.svnrev)
   print("This is ptcut v{}, r{}".format(vsn, r), file=file)
   print("It is now {}".format(time), file=file)
   print("command line: {}".format(" ".join(sys.argv)), file=file)


def main():
   from datetime import datetime
   boot_time = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
   status(boot_time)

   print_out = False
   sorting = 1
   verbose = 0
   models = []
   algo = 3
   like = 14
   global backend
   collect = 0
   skip_formula = []
   common = True
   sumup = False
   epname = "5"
   ep = Integer(1)/Integer(epname)
   rnd = 0
   force = False
   keep_coeff = False
   resort = False
   paramwise = True
   test = False
   convexhull = True
   grid_data = []
   log_file_name = None
   complex = False
   save_solutions = True
   bag_order = []
   fusion = False
   save_solutions_to_log = False
   con_type = 0
   connected_components = False
   multiple_lpfiles = False
   jeff = False
   multi_log = False
   runs = 1
   max_run_time = inf
   log_append = False
   work_dir = ""

   def get_ep(i):
      if i[0] in "pc" and i[1:].isdigit():
         idx = int(i[1:])
         try:
            rep = make_prime(idx) + (1 if i[0] == "c" else 0)
         except IndexError:
            print("No prime with {} digits found".format(idx))
            sys.exit(1)
      else:
         rep = int(i)
      return Integer(1) / Integer(rep), i

   for i in sys.argv[1:]:
      if collect:
         if collect == 1:
            skip_formula.extend([int(j) for j in i.split(",")])
         elif collect == 2:
            global chull_f1
            chull_f1 = bestnum(i)
         elif collect == 3:
            global chull_f2
            chull_f2 = bestnum(i)
         elif collect == 4:
            ep, epname = get_ep(i)
         elif collect == 5:
            global chull_f3
            chull_f3 = bestnum(i)
         elif collect == 6:
            global chull_f4
            chull_f4 = bestnum(i)
         elif collect == 7:
            grid_data.extend(read_grid_data(i))
         elif collect == 8:
            rnd = int(i)
         elif collect == 9:
            log_file_name = i
            multi_log = True
         elif collect == 10:
            bag_order.extend([int(j) for j in i.split(",")])
         elif collect == 11:
            con_type = int(i)
         elif collect == 12:
            runs = int(i)
         elif collect == 13:
            max_run_time = float(i)
         elif collect == 14:
            work_dir = i
         collect = 0
      elif i.startswith("-"):
         if i == "-t":
            runs = 2**63
         elif i == "-p":
            print_out = True
         elif i == "--sortup":
            sorting = 1
         elif i == "--sortdown":
            sorting = 2
         elif i == "--shuffle":
            sorting = 3
         elif i == "--noshuffle":
            sorting = 0
         elif i.startswith("-a"):
            algo = int(i[2:])
         elif i.startswith("-l"):
            like = int(i[2:])
         elif i == "--bp":
            backend = "ppl"
         elif i == "--bc":
            backend = "cdd"
         elif i == "--bn":
            backend = "normaliz"
         elif i == "--bf":
            backend = "field"
         elif i == "-v":
            for j in i[1:]:
               if j != 'v':
                  break
               verbose += 1
         elif i == "--verbose":
            verbose += 1
         elif i == "--simple":
            models.extend(biomd_simple)
         elif i == "--easy":
            models.extend(biomd_easy)
         elif i == "--fast":
            models.extend(biomd_fast)
         elif i == "--slow":
            models.extend(biomd_slow)
         elif i == "--slowhull":
            models.extend(biomd_slowhull)
         elif i == "--skip":
            collect = 1
         elif i == "--common":
            common = True
         elif i == "--nocommon" or i == "--nc":
            common = False
         elif i == "--sum":
            sumup = True
         elif i == "--nosum":
            sumup = False
         elif i == "-f":
            force = True
         elif i == "--keep-coeff":
            keep_coeff = True
         elif i == "--resort":
            resort = True
         elif i == "--merge-param":
            paramwise = False
         elif i == "--test":
            test = True
         elif i == "--nohull" or i == "--nh":
            convexhull = False
         elif i == "--hull":
            convexhull = True
         elif i == "--hull1":
            convexhull = True
            collect = 2
         elif i == "--hull2":
            convexhull = True
            collect = 3
         elif i == "--hull3":
            convexhull = True
            collect = 5
         elif i == "--hull4":
            convexhull = True
            collect = 6
         elif i == "--ep":
            collect = 4
         elif i.startswith("-e"):
            ep, epname = get_ep(i[2:])
         elif i == "--np" or i == "--no-progress":
            global progress
            set_progress(False)
         elif i == "--grid":
            collect = 7
         elif i == "-r" or i == "--round":
            collect = 8
         elif i.startswith("-r"):
            rnd = int(i[2:])
         elif i == "--log":
            collect = 9
         elif i == "--append":
            log_append = True
         elif i == "--complex" or i == "-c":
            complex = True
         elif i == "--nosave":
            save_solutions = False
         elif i == "--order":
            collect = 10
         elif i == "--fusion":
            fusion = True
         elif i == "--nofusion":
            fusion = False
         elif i == "--soltolog" or i == "--stl":
            save_solutions_to_log = True
         elif i == "--contype":
            collect = 11
         elif i == "--concomp" or i == "--cc":
            connected_components = True
         elif i == "--multiple-lpfiles":
            multiple_lpfiles = True
         elif i == "--jeff":
            jeff = True
         elif i == "--runs":
            collect = 12
         elif i == "--maxruntime":
            collect = 13
         elif i == "--dir":
            collect = 14
      else:
         models.append(i)

   time_prec()

   if test:
      sys.exit(1)

   if not models:
      print("Please specify a model")
      sys.exit(1)

   if not common:
      convexhull = False

   if rnd > 0:
      paramwise = False

   if jeff:
      complex = True
      grid_data = []
      sumup = False
      keep_coeff = False
      paramwise = True
      ep = 0
      epname = "x"
      rnd = 0     # dummy

   if grid_data:
      save_solutions = False
      print_out = False

   while runs > 0:
      for mod in models:
         flags = ("s" if sumup else "") + ("k" if keep_coeff else "") + ("" if paramwise or rnd > 0 else "m")
         pflags = flags + ("c" if complex else "")
         sflags = pflags + ("f" if fusion else "")
         mod_dir = (work_dir + os.sep if work_dir else "") + "db{}{}{}".format(os.sep, mod, os.sep)
         fname0 = "{}ep{}{}".format(mod_dir, epname, ("-r{}".format(rnd) if rnd > 0 else ""))
         tfname1 = fname0 + ("-" + flags if flags else "")
         fname1 = fname0 + ("-" + pflags if pflags else "")
         sfname1 = fname0 + ("-" + sflags if sflags else "")

         if not multi_log:
            # each model has its own log file
            if get_log_file():
               get_log_file().close()
               set_log_file(None)
            log_file_name = sfname1 + "-log.txt"
         
         assert log_file_name
         if not get_log_file():
            # log file set (through --log as multi_log or individually), but not yet opened
            set_log_file(open(log_file_name, "a" if log_append else "w"))
            if get_log_file():
               status(boot_time, file=get_log_file())

         prt()
         prt("-" * 70)
         if ep == 0:
            epstr = "x"
         elif 1 / ep < 11000:
            epstr = str(ep)
         else:
            lg = int(floor(log(1/ep, 10)))
            epstr = "1/(10**{}+{})".format(lg, 1/ep - 10**lg)
         tropstr = "" if jeff else "ep={}, round={}, complex={}, sumup={}, keep_coeff={}, paramwise={}, ".format(epstr, rnd, complex, sumup, keep_coeff, paramwise)
         prt("Solving {} with {}fusion={}, algorithm {}, resort={}, common={}, convexhull={}, likeness {}, backend {}".format(
            mod, tropstr, fusion, algo, resort, common, (convexhull, chull_f1, chull_f2, chull_f3, chull_f4), like, backend))
         prt()

         for override in sample_grid(grid_data):
            # if grid sampling, print which parameters we're using
            if grid_data:
               param_str = ", ".join(["{} = {}".format(k, v) for k,v in override])
               prt("\nGrid point: {}".format(param_str))
            # load or compute tropical system
            if jeff:
               ts = load_jeff_system(mod_dir)
            else:
               ts = None if force or grid_data else load_tropical(tfname1)
               if not ts:
                  prt(end="Computing tropical system...", flush=True)
                  ts = tropicalize_system(mod, mod_dir, ep=float(ep), scale=10**rnd, sumup=sumup, verbose=verbose, keep_coeff=keep_coeff, paramwise=paramwise, param_override=dict(override))
                  if not grid_data:
                     save_tropical(ts, tfname1)
            # load solution, if existant
            sol = None
            if not complex and not grid_data and ep == Integer(1) / Integer(5) and rnd == 0 and mod.startswith("BIOMD"):
               try:
                  n = int(mod[5:])
               except ValueError:
                  pass
               else:
                  sol = load_satyas_solution(n)

            pts_bags = None if force or grid_data else load_polytopes(fname1)
            if not pts_bags:
               prt(end="Creating polytopes... ", flush=True)
               pts_bags = make_polytopes(ts, skip_formula, verbose=verbose, complex=complex)
               if not grid_data:
                  save_polytopes(pts_bags, fname1)
            prt()

            if sorting == 1:
               prt("Sorting ascending...")
               pts_bags = sorted(pts_bags, key=lambda x: len(x))
            elif sorting == 2:
               prt("Sorting descending...")
               pts_bags = sorted(pts_bags, key=lambda x: -len(x))
            elif sorting == 3:
               prt("Shuffling...")
               shuffle(pts_bags)

            # if --order was specified, order listed bags to the beginning
            if bag_order:
               bags = []
               for i in bag_order:
                  for j in range(len(pts_bags)):
                     if not pts_bags[j] is None and pts_bags[j].name == "#{}".format(i):
                        bags.append(pts_bags[j])
                        pts_bags[j] = None
               for b in pts_bags:
                  if not b is None:
                     bags.append(b)
               pts_bags = bags
               sorting = 0

            rs, stats = intersect_all(pts_bags, verbose=1, algo=algo, like=like, common=common, sorting=sorting, resort=resort, convexhull=convexhull, fname1=fname1, max_run_time=max_run_time)
            if not stats["aborted"]:
               # fuse polytopes
               if fusion:
                  from fuse import fuse_polytopes
                  ftime = mytime()
                  olen = len(rs)
                  fuse_polytopes(rs, progress=True)
                  ftime = mytime() - ftime
                  if len(rs) != olen:
                     prt("Fusion reduced polytopes from {} to {}.  Fusion time: {:.3f} sec".format(olen, len(rs), ftime))
               # print number of solutions
               s = "Solutions: {}".format(len(rs))
               if rs:
                  s += ", dim={:.1f}, hs={:.1f}, vs={:.1f}".format(avg_dim(rs), avg_planes(rs), avg_vrep(rs))
               s += ", max={}".format(stats["max_pt"])
               prt(s)
               # sort into a distict representation
               prt("Canonicalizing...", flush=True)
               rs = canonicalize(rs)
               # count connected components
               if connected_components:
                  adj, adj_str = build_graph(rs, con_type=con_type, dbg=True)
                  import graph
                  cc = graph.connected_components(adj)
                  prt("Connected components: {}".format(cc))
                  prt("Graph ID: {}".format(graph.graph_name(adj_str)))

               if get_log_file():
                  get_log_file().flush()

               if save_solutions or save_solutions_to_log:
                  prt(end="Saving solutions... ", flush=True)
                  if save_solutions and not multiple_lpfiles or save_solutions_to_log:
                     s = sol_to_string(rs)
                  if save_solutions:
                     if multiple_lpfiles:
                        sol_to_lpfile(rs, fname1)
                     else:
                        sol_string_to_one_lpfile(s, sfname1)
                  if save_solutions_to_log:
                     if get_log_file():
                        get_log_file().write(s)
                     else:
                        prt(end=s)
                  prt(flush=True)
                  save_solutions = False        # just do it once on multiple runs

               if sol:
                  notfound = compare_solutions(rs, sol)
                  if notfound == 0:
                     prt("Solutions match Satya's solutions")
                  else:
                     prt("Solutions have {} differences from Satya's solutions".format(notfound))

               if print_out:
                  prt()
                  prt("Solution:")
                  if not rs:
                     prt("Solution set is empty")
                  #for i in rs:
                  #   prt(i.Hrepresentation())
                  for i in rs:
                     prt("   {}".format(i.Vrepresentation()))
                  if sol:
                     prt()
                     prt("Satya's solution is:")
                     for i in sol:
                        prt("   {}".format(i.Vrepresentation()))

               prt("Calculation done.")

      runs -= 1


if __name__ == "__main__":
   import doctest
   doctest.testmod(verbose=False)
   import biomd
   doctest.testmod(biomd, verbose=False)
   import util
   doctest.testmod(util, verbose=False)
   import fuse
   doctest.testmod(fuse, verbose=False)
   import graph
   doctest.testmod(graph, verbose=False)

   main()
